/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_close_endpoint.c,v 1.91 2006/05/29 22:54:30 loic Exp $";

/**
   @file mx_close_endpoint.c
   The file containing the MX API function mx_close_endpoint().
   mx_close_endpoint() closes an opened MX endpoint.
*/

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx__internals.h"
#  include "mx__partner.h"
#include "mx__endpoint.h"
#include "mx__regcache.h"
#include "mx_connect.h"
#include "mx__driver_interface.h"
#include "mx__lib.h"
#include "mx__handle_map.h"
#include "mx__request.h"
#include "mx__stack.h"
#include "mx__fops.h"
#include "mx__error.h"
#include "mx__shmem.h"
#include "mx__debug_dump.h"

MX_FUNC(mx_return_t)
mx_close_endpoint (mx_endpoint_t endpoint)
{
  mx_return_t ret = MX_SUCCESS;
  int init_count;
  mx_endpoint_t *prev;
  uint32_t i;

  MX__MUTEX_LOCK(MX_LOCK);
  init_count = Mx_init_count;

  if (init_count <= 0) {
    ret = MX_NOT_INITIALIZED;
    goto abort_with_nothing;
  }

  for (prev = &Mx_endpoints; *prev; prev = &(*prev)->next) {
    if (*prev == endpoint)
      break;
  }
  if (endpoint != *prev) {
    ret = MX_BAD_ENDPOINT;
    goto abort_with_nothing;
  }

  if (endpoint->in_handler) {
    ret = MX_CLOSE_IN_HANDLER;
    goto abort_with_nothing;
  }

  endpoint->cancelled = 1;

  if (mx__opt.stats) {
    mx__dump_endpoint_stats(endpoint);
    if (mx__opt.stats > 1)
      mx__dump_endpoint(endpoint, mx__opt.stats-1);
  }
  mx__end_connect(endpoint);
  if (endpoint->application_rank_info) {
    mx_free(endpoint->application_rank_info);
  }
#if MX_THREAD_SAFE
  /* wakeup the progression thread so he knows he need to exit */
  mx__wake(endpoint->handle);
#endif
  if (!mx__opt.monothread) {
    MX__THREAD_JOIN(&endpoint->thread);
  }

  mx__finish_buffered_requests(endpoint);
  MX__MUTEX_LOCK(&Mx_rcache_lock);
  *prev = endpoint->next;
  endpoint->next = NULL;
  MX__MUTEX_UNLOCK(&Mx_rcache_lock);

  MX__MUTEX_DESTROY(&endpoint->lock);
  MX__EVENT_DESTROY(&endpoint->in_handler_event);
  mx_rdmawin_finalize(endpoint);
#if MX_USE_SHMEM
  if (!mx__opt.disable_shmem) {
    mx__shm_close(endpoint, endpoint->shm->shmq, endpoint->myself->eid, 1, 0);
    for (i = 0; i < endpoint->max_endpoints; ++i) {
      struct mx__shm_peer *peer = &endpoint->shm->peers[i];
      if (peer->shm_snd_fifo) {
	mx__shm_close(endpoint, peer->shm_snd_fifo, i, 1, 1);
      }
    }
    mx_free(endpoint->shm->peers);
    mx_free(endpoint->shm);
  }
#endif
  for (i = 0; i < endpoint->max_peers * endpoint->max_endpoints; ++i) {
    mx_free (endpoint->remote_ep[i]);
  }
  mx_free(endpoint->remote_ep);
  mx_free(endpoint->ctxid);
  mx__hm_fini(endpoint->handle_map);
  mx__rl_fini(&endpoint->req_lookaside);
  if (endpoint->send_pool)
    mx_free(endpoint->send_pool);
  if (endpoint->small_msg_ptrs)
    mx__ptr_stack_dealloc(endpoint->small_msg_ptrs);
  mx_free(endpoint->req_ring);
  mx__unmap(endpoint);
  if (endpoint->rdma_requests)
    mx_free(endpoint->rdma_requests);
  mx_rdma_finalize(&endpoint->rdmas);
  if (endpoint->handle != MX__INVALID_HANDLE)
    mx__close(endpoint->handle);

  mx_free(endpoint);

  MX__MUTEX_UNLOCK(MX_LOCK);
  return ret;

 abort_with_nothing:
  MX__MUTEX_UNLOCK(MX_LOCK);
  return mx__error(endpoint, "mx_close_endpoint",ret);
}
